<?php
namespace Controllers;

use Config\Database;
use Models\User;
use Core\Response;
use Core\Request;
use Core\JwtHandler;
use Core\Auth;

class AuthController {
    private $db;
    private $user;

    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->user = new User($this->db);
    }

    public function register() {
        $data = Request::getJsonBody();

        if (!empty($data->name) && !empty($data->email) && !empty($data->password) && !empty($data->phone)) {
            $this->user->name = $data->name;
            $this->user->email = $data->email;
            $this->user->password = $data->password;
            $this->user->phone = $data->phone;

            if ($this->user->emailExists()) {
                Response::error("Email already exists");
            }

            if ($this->user->create()) {
                Response::created(["message" => "User registered successfully"]);
            } else {
                Response::error("Unable to register user", 503);
            }
        } else {
            Response::validationError();
        }
    }

    public function login() {
        $data = Request::getJsonBody();

        if (!empty($data->email) && !empty($data->password)) {
            $this->user->email = $data->email;

            if ($this->user->emailExists()) {
                if (password_verify($data->password, $this->user->password)) {
                    $jwt = new JwtHandler($this->db);
                    $token = $jwt->generateToken($this->user->id, $this->user->role);

                    Response::json([
                        "message" => "Login successful",
                        "token" => $token,
                        "user" => [
                            "id" => $this->user->id,
                            "name" => $this->user->name,
                            "role" => $this->user->role
                        ]
                    ]);
                } else {
                    Response::unauthorized("Invalid password");
                }
            } else {
                Response::notFound("User not found");
            }
        } else {
            Response::validationError();
        }
    }
    
    public function logout() {
        $headers = apache_request_headers();
        $token = null;
        if (isset($headers['Authorization'])) {
            $matches = [];
            if (preg_match('/Bearer\s(\S+)/', $headers['Authorization'], $matches)) {
                $token = $matches[1];
            }
        }
        
        if($token) {
            $jwt = new JwtHandler($this->db);
            if($jwt->blacklistToken($token)) {
                Response::json(["message" => "Logout successful"]);
            } else {
                Response::error("Logout failed");
            }
        } else {
            Response::error("No token provided");
        }
    }

    public function getProfile() {
        $userData = Auth::validate();
        $this->user->id = $userData->id;
        $profile = $this->user->getProfile();
        if($profile) {
            Response::json($profile);
        } else {
            Response::notFound("User not found");
        }
    }
    
    public function updateProfile() {
        $userData = Auth::validate();
        $data = Request::getJsonBody();
        
        if(!empty($data->name) && !empty($data->phone)) {
            $this->user->id = $userData->id;
            $this->user->name = $data->name;
            $this->user->phone = $data->phone;
            
            if($this->user->updateProfile()) {
                Response::json(["message" => "Profile updated successfully"]);
            } else {
                Response::error("Unable to update profile", 503);
            }
        } else {
            Response::validationError();
        }
    }
}
