<?php
namespace Controllers;

use Config\Database;
use Models\CustomerRequest;
use Core\Response;
use Core\Request;
use Core\Auth;

class RequestController {
    private $db;
    private $request;

    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
        $this->request = new CustomerRequest($this->db);
    }

    public function create() {
        $user = Auth::validate();
        $data = Request::getJsonBody();

        if (!empty($data->service_id)) {
            $this->request->user_id = $user->id;
            $this->request->service_id = $data->service_id;
            $this->request->notes = $data->notes ?? '';

            if ($this->request->create()) {
                Response::created(["message" => "Request created successfully", "id" => $this->request->id]);
            } else {
                Response::error("Unable to create request", 503);
            }
        } else {
            Response::validationError();
        }
    }

    public function myRequests() {
        $user = Auth::validate();
        $this->request->user_id = $user->id;
        $stmt = $this->request->readByUser();
        $requests = $stmt->fetchAll(\PDO::FETCH_ASSOC);
        Response::json($requests);
    }

    public function show($id) {
        $user = Auth::validate();
        $this->request->id = $id;
        $request = $this->request->readOne();
        
        if ($request) {
            if ($user->role !== 'admin' && $request['user_id'] != $user->id) {
                Response::error("Access denied", 403);
            }
            $documents = $this->request->getDocuments();
            $request['documents'] = $documents;
            Response::json($request);
        } else {
            Response::notFound("Request not found");
        }
    }

    public function update($id) {
        $user = Auth::validate();
        if ($user->role !== 'admin') {
            Response::error("Access denied", 403);
        }

        $data = Request::getJsonBody();

        if (!empty($data->status)) {
            $this->request->id = $id;
            $this->request->status = $data->status;

            if ($this->request->updateStatus()) {
                Response::json(["message" => "Request status updated successfully"]);
            } else {
                Response::error("Unable to update request", 503);
            }
        } else {
            Response::validationError();
        }
    }

    public function uploadDocument($id) {
        $user = Auth::validate();
        $this->request->id = $id;
        $request = $this->request->readOne();

        if (!$request) {
            Response::error("Request not found", 404);
        }

        if ($user->role !== 'admin' && $request['user_id'] != $user->id) {
            Response::error("Access denied", 403);
        }

        if (isset($_FILES['document'])) {
            $target_dir = __DIR__ . "/../uploads/";
            if (!file_exists($target_dir)) {
                mkdir($target_dir, 0777, true);
            }
            
            $file_name = time() . "_" . basename($_FILES["document"]["name"]);
            $target_file = $target_dir . $file_name;
            
            if (move_uploaded_file($_FILES["document"]["tmp_name"], $target_file)) {
                $this->request->id = $id;
                $this->request->document_name = $_FILES["document"]["name"];
                $this->request->document_url = "uploads/" . $file_name;
                
                if ($this->request->addDocument()) {
                    Response::json(["message" => "Document uploaded successfully", "url" => $this->request->document_url]);
                } else {
                    Response::error("Database error saving document info", 503);
                }
            } else {
                Response::error("Error uploading file");
            }
        } else {
            Response::error("No file uploaded");
        }
    }
}
