<?php
namespace Core;

use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use PDO;

class JwtHandler {
    private $secret;
    private $db;

    public function __construct($db) {
        $this->secret = $_ENV['JWT_SECRET'];
        $this->db = $db;
    }

    public function generateToken($user_id, $role) {
        $payload = [
            'iss' => "uae-docs-api",
            'aud' => "uae-docs-users",
            'iat' => time(),
            'exp' => time() + (60 * 60 * 24), // 1 day
            'data' => [
                'id' => $user_id,
                'role' => $role
            ]
        ];

        return JWT::encode($payload, $this->secret, 'HS256');
    }

    public function validateToken($token) {
        try {
            // Check blacklist
            $query = "SELECT id FROM jwt_blacklist WHERE token = :token";
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(":token", $token);
            $stmt->execute();
            if ($stmt->rowCount() > 0) {
                return false;
            }

            $decoded = JWT::decode($token, new Key($this->secret, 'HS256'));
            return $decoded->data;
        } catch (\Exception $e) {
            // Log error if needed: error_log($e->getMessage());
            return false;
        }
    }
    
    public function blacklistToken($token) {
        try {
            // Decode without validation to get expiry
            // We suppress errors because if it's invalid we can't get exp anyway, 
            // but we should still blacklist it if possible or just ignore.
            // However, for logout, the token should be valid.
            // If it's already expired, no need to blacklist, but let's be safe.
            $tks = explode('.', $token);
            if (count($tks) != 3) {
                return false;
            }
            
            $body = json_decode(base64_decode(str_replace(['-', '_'], ['+', '/'], $tks[1])));
            if (!isset($body->exp)) {
                $expiry = date('Y-m-d H:i:s', time() + 86400);
            } else {
                $expiry = date('Y-m-d H:i:s', $body->exp);
            }
             
            $query = "INSERT INTO jwt_blacklist (token, expiry_date) VALUES (:token, :expiry)";
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(":token", $token);
            $stmt->bindParam(":expiry", $expiry);
            return $stmt->execute();
        } catch (\Exception $e) {
            return false;
        }
    }
}
