const API_URL = 'http://localhost/doc_process/uae-docs-api/public/index.php'; // Adjust if needed

document.addEventListener('DOMContentLoaded', () => {
    checkAuth();
    setupEventListeners();
});

function setupEventListeners() {
    // Auth Tabs
    document.getElementById('loginTab').addEventListener('click', (e) => {
        e.preventDefault();
        document.getElementById('loginTab').classList.add('active');
        document.getElementById('registerTab').classList.remove('active');
        document.getElementById('loginForm').classList.remove('d-none');
        document.getElementById('registerForm').classList.add('d-none');
    });

    document.getElementById('registerTab').addEventListener('click', (e) => {
        e.preventDefault();
        document.getElementById('registerTab').classList.add('active');
        document.getElementById('loginTab').classList.remove('active');
        document.getElementById('registerForm').classList.remove('d-none');
        document.getElementById('loginForm').classList.add('d-none');
    });

    // Forms
    document.getElementById('loginForm').addEventListener('submit', handleLogin);
    document.getElementById('registerForm').addEventListener('submit', handleRegister);
    document.getElementById('createRequestForm').addEventListener('submit', handleCreateRequest);
}

function checkAuth() {
    const token = localStorage.getItem('token');
    const navLinks = document.getElementById('navLinks');
    const authSection = document.getElementById('authSection');
    const dashboardSection = document.getElementById('dashboardSection');
    const requestsSection = document.getElementById('requestsSection');

    if (token) {
        // Logged in
        navLinks.innerHTML = `
            <li class="nav-item"><a class="nav-link" href="#" onclick="showDashboard()">Services</a></li>
            <li class="nav-item"><a class="nav-link" href="#" onclick="showRequests()">My Requests</a></li>
            <li class="nav-item"><a class="nav-link" href="#" onclick="logout()">Logout</a></li>
        `;
        authSection.classList.add('d-none');
        showDashboard();
    } else {
        // Not logged in
        navLinks.innerHTML = `
            <li class="nav-item"><a class="nav-link" href="#" onclick="showAuth()">Login/Register</a></li>
        `;
        authSection.classList.remove('d-none');
        dashboardSection.classList.add('d-none');
        requestsSection.classList.add('d-none');
    }
}

function showAuth() {
    document.getElementById('authSection').classList.remove('d-none');
    document.getElementById('dashboardSection').classList.add('d-none');
    document.getElementById('requestsSection').classList.add('d-none');
}

function showDashboard() {
    document.getElementById('authSection').classList.add('d-none');
    document.getElementById('dashboardSection').classList.remove('d-none');
    document.getElementById('requestsSection').classList.add('d-none');
    loadServices();
}

function showRequests() {
    document.getElementById('authSection').classList.add('d-none');
    document.getElementById('dashboardSection').classList.add('d-none');
    document.getElementById('requestsSection').classList.remove('d-none');
    loadRequests();
}

async function handleLogin(e) {
    e.preventDefault();
    const formData = new FormData(e.target);
    const data = Object.fromEntries(formData.entries());

    try {
        const response = await fetch(`${API_URL}/auth/login`, {
            method: 'POST',
            body: JSON.stringify(data)
        });
        const result = await response.json();

        if (response.ok) {
            localStorage.setItem('token', result.token);
            localStorage.setItem('user', JSON.stringify(result.user));
            checkAuth();
            showAlert('Login successful', 'success');
        } else {
            showAlert(result.error || 'Login failed', 'danger');
        }
    } catch (error) {
        showAlert('An error occurred', 'danger');
    }
}

async function handleRegister(e) {
    e.preventDefault();
    const formData = new FormData(e.target);
    const data = Object.fromEntries(formData.entries());

    try {
        const response = await fetch(`${API_URL}/auth/register`, {
            method: 'POST',
            body: JSON.stringify(data)
        });
        const result = await response.json();

        if (response.ok) {
            showAlert('Registration successful. Please login.', 'success');
            document.getElementById('loginTab').click();
        } else {
            showAlert(result.error || 'Registration failed', 'danger');
        }
    } catch (error) {
        showAlert('An error occurred', 'danger');
    }
}

function logout() {
    localStorage.removeItem('token');
    localStorage.removeItem('user');
    checkAuth();
}

async function loadServices() {
    try {
        const response = await fetch(`${API_URL}/services`);
        const services = await response.json();
        const list = document.getElementById('servicesList');

        if (services.length === 0) {
            list.innerHTML = '<p>No services available.</p>';
            return;
        }

        list.innerHTML = services.map(service => `
            <div class="col-md-4 mb-4">
                <div class="card service-card h-100">
                    <div class="card-body">
                        <h5 class="card-title">${service.name}</h5>
                        <h6 class="card-subtitle mb-2 text-muted">${service.category_name || 'General'}</h6>
                        <p class="card-text">${service.description || 'No description'}</p>
                        <p class="card-text"><strong>Price:</strong> AED ${service.base_price}</p>
                        <button class="btn btn-primary" onclick="openRequestModal(${service.id}, '${service.name}', ${service.base_price})">Request Service</button>
                    </div>
                </div>
            </div>
        `).join('');
    } catch (error) {
        console.error('Error loading services:', error);
    }
}

function openRequestModal(id, name, price) {
    document.getElementById('modalServiceId').value = id;
    document.getElementById('modalServiceName').value = name;
    document.getElementById('modalServicePrice').value = price;
    new bootstrap.Modal(document.getElementById('requestModal')).show();
}

async function handleCreateRequest(e) {
    e.preventDefault();
    const formData = new FormData(e.target);
    const data = Object.fromEntries(formData.entries());
    const token = localStorage.getItem('token');

    try {
        const response = await fetch(`${API_URL}/requests/create`, {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify(data)
        });
        const result = await response.json();

        if (response.ok) {
            showAlert('Request created successfully', 'success');
            bootstrap.Modal.getInstance(document.getElementById('requestModal')).hide();
            showRequests();
        } else {
            showAlert(result.error || 'Failed to create request', 'danger');
        }
    } catch (error) {
        showAlert('An error occurred', 'danger');
    }
}

async function loadRequests() {
    const token = localStorage.getItem('token');
    try {
        const response = await fetch(`${API_URL}/requests/my`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        const requests = await response.json();
        const tbody = document.getElementById('requestsTableBody');

        if (requests.length === 0) {
            tbody.innerHTML = '<tr><td colspan="5">No requests found.</td></tr>';
            return;
        }

        tbody.innerHTML = requests.map(req => `
            <tr>
                <td>${req.id}</td>
                <td>${req.service_name}</td>
                <td><span class="badge bg-${getStatusColor(req.status)}">${req.status}</span></td>
                <td>${new Date(req.created_at).toLocaleDateString()}</td>
                <td>
                    <button class="btn btn-sm btn-info" onclick="viewRequest(${req.id})">View</button>
                </td>
            </tr>
        `).join('');
    } catch (error) {
        console.error('Error loading requests:', error);
    }
}

function getStatusColor(status) {
    switch (status) {
        case 'pending': return 'warning';
        case 'processing': return 'info';
        case 'completed': return 'success';
        case 'rejected': return 'danger';
        default: return 'secondary';
    }
}

function showAlert(message, type) {
    const container = document.getElementById('alertContainer');
    container.innerHTML = `
        <div class="alert alert-${type} alert-dismissible fade show" role="alert">
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    `;
}

function viewRequest(id) {
    // Implement view request details logic if needed
    document.getElementById('uploadForm').addEventListener('submit', handleUpload);
    document.getElementById('paymentForm').addEventListener('submit', handlePayment);
}

// ... (keep existing code) ...

function openUploadModal(id) {
    document.getElementById('uploadRequestId').value = id;
    new bootstrap.Modal(document.getElementById('uploadModal')).show();
}

function openPaymentModal(id, amount) {
    document.getElementById('paymentRequestId').value = id;
    document.getElementById('paymentAmount').value = amount;
    new bootstrap.Modal(document.getElementById('paymentModal')).show();
}

async function handleUpload(e) {
    e.preventDefault();
    const formData = new FormData(e.target);
    const requestId = document.getElementById('uploadRequestId').value;
    const token = localStorage.getItem('token');

    try {
        const response = await fetch(`${API_URL}/requests/${requestId}/upload-document`, {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${token}`
            },
            body: formData // FormData handles multipart/form-data automatically
        });
        const result = await response.json();

        if (response.ok) {
            showAlert('Document uploaded successfully', 'success');
            bootstrap.Modal.getInstance(document.getElementById('uploadModal')).hide();
            showRequests(); // Refresh list
        } else {
            showAlert(result.error || 'Upload failed', 'danger');
        }
    } catch (error) {
        showAlert('An error occurred', 'danger');
    }
}

async function handlePayment(e) {
    e.preventDefault();
    const formData = new FormData(e.target);
    const data = Object.fromEntries(formData.entries());
    const token = localStorage.getItem('token');

    try {
        const response = await fetch(`${API_URL}/payments/initiate`, {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${token}`,
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(data)
        });
        const result = await response.json();

        if (response.ok) {
            showAlert('Payment initiated successfully. Transaction ID: ' + result.transaction_id, 'success');
            bootstrap.Modal.getInstance(document.getElementById('paymentModal')).hide();
            // Ideally, redirect to payment gateway or show success
        } else {
            showAlert(result.error || 'Payment failed', 'danger');
        }
    } catch (error) {
        showAlert('An error occurred', 'danger');
    }
}

// Update loadRequests to show buttons
async function loadRequests() {
    const token = localStorage.getItem('token');
    try {
        const response = await fetch(`${API_URL}/requests/my`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        const requests = await response.json();
        const tbody = document.getElementById('requestsTableBody');

        if (requests.length === 0) {
            tbody.innerHTML = '<tr><td colspan="5">No requests found.</td></tr>';
            return;
        }

        tbody.innerHTML = requests.map(req => `
            <tr>
                <td>${req.id}</td>
                <td>${req.service_name}</td>
                <td><span class="badge bg-${getStatusColor(req.status)}">${req.status}</span></td>
                <td>${new Date(req.created_at).toLocaleDateString()}</td>
                <td>
                    <button class="btn btn-sm btn-info" onclick="viewRequest(${req.id})">View</button>
                    <button class="btn btn-sm btn-secondary" onclick="openUploadModal(${req.id})">Upload</button>
                    <button class="btn btn-sm btn-success" onclick="openPaymentModal(${req.id}, ${req.base_price})">Pay</button>
                </td>
            </tr>
        `).join('');
    } catch (error) {
        console.error('Error loading requests:', error);
    }
}

function viewRequest(id) {
    // Fetch details including documents
    const token = localStorage.getItem('token');
    fetch(`${API_URL}/requests/${id}`, {
        headers: { 'Authorization': `Bearer ${token}` }
    })
        .then(res => res.json())
        .then(data => {
            let docsHtml = '';
            if (data.documents && data.documents.length > 0) {
                docsHtml = '<h6>Documents:</h6><ul>' +
                    data.documents.map(d => `<li><a href="${d.document_url}" target="_blank">${d.document_name}</a></li>`).join('') +
                    '</ul>';
            } else {
                docsHtml = '<p>No documents uploaded.</p>';
            }

            const modalHtml = `
            <div class="modal fade" id="viewRequestModal" tabindex="-1">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">Request #${data.id}</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <p><strong>Service:</strong> ${data.service_name}</p>
                            <p><strong>Status:</strong> ${data.status}</p>
                            <p><strong>Notes:</strong> ${data.notes || 'None'}</p>
                            ${docsHtml}
                        </div>
                    </div>
                </div>
            </div>
        `;

            // Remove existing modal if any
            const existingModal = document.getElementById('viewRequestModal');
            if (existingModal) existingModal.remove();

            document.body.insertAdjacentHTML('beforeend', modalHtml);
            new bootstrap.Modal(document.getElementById('viewRequestModal')).show();
        })
        .catch(err => showAlert('Error fetching details', 'danger'));
}
